<!DOCTYPE html>
<html lang="es">
<head>
    <meta charset="utf-8">
    <meta name="viewport" content="width=device-width, initial-scale=1">
    <title>Gestión de Clientes - Bar VIP Admin</title>
    
    <!-- CSS Principal -->
    <link rel="stylesheet" href="assets/css/admin.css">
    
    <!-- DataTables CSS -->
    <link rel="stylesheet" href="https://cdn.datatables.net/1.13.7/css/jquery.dataTables.min.css">
    <link rel="stylesheet" href="https://cdn.datatables.net/responsive/2.5.0/css/responsive.dataTables.min.css">
    
    <!-- Font Awesome para iconos -->
    <link rel="stylesheet" href="https://cdnjs.cloudflare.com/ajax/libs/font-awesome/6.4.0/css/all.min.css">
    
    <!-- jQuery y DataTables JS -->
    <script src="https://code.jquery.com/jquery-3.7.1.min.js"></script>
    <script src="https://cdn.datatables.net/1.13.7/js/jquery.dataTables.min.js"></script>
    <script src="https://cdn.datatables.net/responsive/2.5.0/js/dataTables.responsive.min.js"></script>
</head>
<body>
    <!-- Sidebar -->
    <nav class="sidebar" id="sidebar">
        <div class="sidebar-header">
            <div class="logo">
                <i class="fas fa-cocktail"></i>
                <span>Bar VIP Admin</span>
            </div>
            <button class="sidebar-toggle" id="sidebarToggle">
                <i class="fas fa-bars"></i>
            </button>
        </div>
        
        <ul class="sidebar-nav">
            <li class="nav-item">
                <a href="index.php" class="nav-link">
                    <i class="fas fa-chart-line"></i>
                    <span>Dashboard</span>
                </a>
            </li>
            <li class="nav-item active">
                <a href="clientes.php" class="nav-link">
                    <i class="fas fa-users"></i>
                    <span>Clientes</span>
                    <span class="badge" id="clientesBadge">0</span>
                </a>
            </li>
            <li class="nav-item">
                <a href="validaciones.php" class="nav-link">
                    <i class="fas fa-user-check"></i>
                    <span>Validaciones</span>
                </a>
            </li>
            <li class="nav-item">
                <a href="reportes.php" class="nav-link">
                    <i class="fas fa-chart-bar"></i>
                    <span>Reportes</span>
                </a>
            </li>
            <li class="nav-item">
                <a href="imagenes.php" class="nav-link">
                    <i class="fas fa-images"></i>
                    <span>Galería</span>
                </a>
            </li>
            <li class="nav-item">
                <a href="configuracion.php" class="nav-link">
                    <i class="fas fa-cog"></i>
                    <span>Configuración</span>
                </a>
            </li>
            <li class="nav-item">
                <a href="logs.php" class="nav-link">
                    <i class="fas fa-list-alt"></i>
                    <span>Logs</span>
                </a>
            </li>
        </ul>
        
        <div class="sidebar-footer">
            <div class="admin-info">
                <i class="fas fa-user-circle"></i>
                <div>
                    <div class="admin-name" id="adminName">Administrador</div>
                    <div class="admin-role">Super Admin</div>
                </div>
            </div>
            <a href="logout.php" class="logout-btn">
                <i class="fas fa-sign-out-alt"></i>
                <span>Cerrar Sesión</span>
            </a>
        </div>
    </nav>

    <!-- Main Content -->
    <div class="main-content" id="mainContent">
        <!-- Top Bar -->
        <header class="topbar">
            <div class="topbar-left">
                <button class="mobile-toggle" id="mobileToggle">
                    <i class="fas fa-bars"></i>
                </button>
                <div class="breadcrumb">
                    <i class="fas fa-users"></i>
                    <span>Gestión de Clientes</span>
                </div>
            </div>
            <div class="topbar-right">
                <div class="search-box">
                    <i class="fas fa-search"></i>
                    <input type="text" placeholder="Buscar cliente..." id="quickSearch">
                </div>
                <button class="refresh-btn" id="refreshData" title="Actualizar datos">
                    <i class="fas fa-sync"></i>
                </button>
                <div class="admin-dropdown">
                    <img src="assets/images/admin-avatar.png" alt="Admin" class="admin-avatar">
                    <i class="fas fa-chevron-down"></i>
                </div>
            </div>
        </header>

        <!-- Content -->
        <main class="content">
            <!-- Action Bar -->
            <div class="action-bar">
                <div class="action-bar-left">
                    <h2 class="page-title">
                        <i class="fas fa-users"></i>
                        Gestión de Clientes
                    </h2>
                    <div class="page-subtitle">
                        Administra todos los clientes registrados en el sistema
                    </div>
                </div>
                <div class="action-bar-right">
                    <button class="btn btn-success" id="btnNuevoCliente">
                        <i class="fas fa-user-plus"></i>
                        Nuevo Cliente
                    </button>
                    <div class="btn-group">
                        <button class="btn btn-outline-primary" id="btnImportar">
                            <i class="fas fa-upload"></i>
                            Importar
                        </button>
                        <button class="btn btn-outline-primary" id="btnExportar">
                            <i class="fas fa-download"></i>
                            Exportar
                        </button>
                    </div>
                    <button class="btn btn-outline-danger" id="btnLimpiarSeleccionados">
                        <i class="fas fa-trash"></i>
                        Eliminar Selec.
                    </button>
                </div>
            </div>

            <!-- Stats Cards -->
            <div class="stats-row">
                <div class="stat-card primary">
                    <div class="stat-icon">
                        <i class="fas fa-users"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-number" id="totalClientes">0</div>
                        <div class="stat-label">Total Clientes</div>
                    </div>
                </div>
                
                <div class="stat-card success">
                    <div class="stat-icon">
                        <i class="fas fa-user-plus"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-number" id="clientesHoy">0</div>
                        <div class="stat-label">Registrados Hoy</div>
                    </div>
                </div>
                
                <div class="stat-card info">
                    <div class="stat-icon">
                        <i class="fas fa-user-check"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-number" id="clientesActivos">0</div>
                        <div class="stat-label">Activos (30 días)</div>
                    </div>
                </div>
                
                <div class="stat-card warning">
                    <div class="stat-icon">
                        <i class="fas fa-exclamation-triangle"></i>
                    </div>
                    <div class="stat-content">
                        <div class="stat-number" id="clientesDuplicados">0</div>
                        <div class="stat-label">Posibles Duplicados</div>
                    </div>
                </div>
            </div>

            <!-- Filters Bar -->
            <div class="filters-bar">
                <div class="filters-left">
                    <div class="filter-group">
                        <label>Período:</label>
                        <select id="filtroFecha" class="form-select">
                            <option value="">Todos</option>
                            <option value="hoy">Hoy</option>
                            <option value="semana">Esta semana</option>
                            <option value="mes">Este mes</option>
                            <option value="3meses">Últimos 3 meses</option>
                            <option value="ano">Este año</option>
                        </select>
                    </div>
                    
                    <div class="filter-group">
                        <label>Estado:</label>
                        <select id="filtroEstado" class="form-select">
                            <option value="">Todos</option>
                            <option value="activo">Activos</option>
                            <option value="inactivo">Inactivos</option>
                            <option value="duplicado">Duplicados</option>
                        </select>
                    </div>
                    
                    <div class="filter-group">
                        <label>Ordenar por:</label>
                        <select id="ordenarPor" class="form-select">
                            <option value="creado_desc">Más recientes</option>
                            <option value="creado_asc">Más antiguos</option>
                            <option value="nombre_asc">Nombre A-Z</option>
                            <option value="nombre_desc">Nombre Z-A</option>
                        </select>
                    </div>
                </div>
                
                <div class="filters-right">
                    <button class="btn btn-outline-secondary" id="btnLimpiarFiltros">
                        <i class="fas fa-times"></i>
                        Limpiar Filtros
                    </button>
                    <div class="view-toggle">
                        <button class="btn btn-sm active" id="vistaTabla" data-view="table">
                            <i class="fas fa-list"></i>
                        </button>
                        <button class="btn btn-sm" id="vistaTarjetas" data-view="cards">
                            <i class="fas fa-th"></i>
                        </button>
                    </div>
                </div>
            </div>

            <!-- Main Table -->
            <div class="table-container">
                <table id="clientesTable" class="display responsive nowrap" style="width:100%">
                    <thead>
                        <tr>
                            <th>
                                <input type="checkbox" id="selectAll">
                            </th>
                            <th>Foto</th>
                            <th>Nombre</th>
                            <th>Email</th>
                            <th>Teléfono</th>
                            <th>Registro</th>
                            <th>Última Validación</th>
                            <th>Estado</th>
                            <th>Acciones</th>
                        </tr>
                    </thead>
                    <tbody>
                        <!-- Los datos se cargan dinámicamente -->
                    </tbody>
                </table>
            </div>

            <!-- Cards View (hidden by default) -->
            <div class="cards-container" id="clientesCards" style="display: none;">
                <!-- Las tarjetas se generan dinámicamente -->
            </div>
        </main>
    </div>

    <!-- Modal Cliente (Ver/Editar) -->
    <div class="modal-overlay" id="clienteModal">
        <div class="modal-dialog modal-large">
            <div class="modal-content">
                <button class="modal-close" id="closeClienteModal">
                    <i class="fas fa-times"></i>
                </button>
                
                <div class="modal-header">
                    <h3 id="modalTitle">Detalles del Cliente</h3>
                </div>
                
                <div class="modal-body">
                    <div class="cliente-details">
                        <!-- Foto del cliente -->
                        <div class="cliente-photo">
                            <div class="photo-container">
                                <img id="clienteFoto" src="" alt="Foto del cliente" style="display: none;">
                                <div class="photo-placeholder" id="photoPlaceholder">
                                    <i class="fas fa-user"></i>
                                </div>
                            </div>
                            <div class="photo-info">
                                <div class="photo-date" id="photoDate"></div>
                                <button class="btn btn-sm btn-outline-primary" id="btnVerFotoCompleta">
                                    <i class="fas fa-expand"></i>
                                    Ver completa
                                </button>
                            </div>
                        </div>
                        
                        <!-- Información del cliente -->
                        <div class="cliente-info">
                            <form id="clienteForm">
                                <input type="hidden" id="clienteId">
                                
                                <div class="form-row">
                                    <div class="form-group">
                                        <label for="clienteNombre">Nombre Completo *</label>
                                        <input type="text" id="clienteNombre" class="form-control" required>
                                    </div>
                                </div>
                                
                                <div class="form-row">
                                    <div class="form-group">
                                        <label for="clienteEmail">Email</label>
                                        <input type="email" id="clienteEmail" class="form-control">
                                    </div>
                                    <div class="form-group">
                                        <label for="clienteTelefono">Teléfono</label>
                                        <input type="tel" id="clienteTelefono" class="form-control">
                                    </div>
                                </div>
                                
                                <div class="form-row">
                                    <div class="form-group">
                                        <label for="clienteRegistro">Fecha de Registro</label>
                                        <input type="text" id="clienteRegistro" class="form-control" readonly>
                                    </div>
                                    <div class="form-group">
                                        <label for="clienteEstado">Estado</label>
                                        <select id="clienteEstado" class="form-control">
                                            <option value="activo">Activo</option>
                                            <option value="inactivo">Inactivo</option>
                                            <option value="bloqueado">Bloqueado</option>
                                        </select>
                                    </div>
                                </div>
                            </form>
                        </div>
                    </div>
                    
                    <!-- Historial de validaciones -->
                    <div class="cliente-history">
                        <h4><i class="fas fa-history"></i> Historial de Validaciones</h4>
                        <div class="history-container" id="historialValidaciones">
                            <!-- Se llena dinámicamente -->
                        </div>
                    </div>
                </div>
                
                <div class="modal-footer">
                    <button class="btn btn-danger" id="btnEliminarCliente">
                        <i class="fas fa-trash"></i>
                        Eliminar
                    </button>
                    <button class="btn btn-secondary" id="btnCancelar">
                        Cancelar
                    </button>
                    <button class="btn btn-primary" id="btnGuardarCliente">
                        <i class="fas fa-save"></i>
                        Guardar Cambios
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Modal Nuevo Cliente -->
    <div class="modal-overlay" id="nuevoClienteModal">
        <div class="modal-dialog modal-medium">
            <div class="modal-content">
                <button class="modal-close" id="closeNuevoModal">
                    <i class="fas fa-times"></i>
                </button>
                
                <div class="modal-header">
                    <h3>Agregar Nuevo Cliente</h3>
                </div>
                
                <div class="modal-body">
                    <form id="nuevoClienteForm" enctype="multipart/form-data">
                        <div class="form-group">
                            <label for="nuevoNombre">Nombre Completo *</label>
                            <input type="text" id="nuevoNombre" class="form-control" required>
                        </div>
                        
                        <div class="form-row">
                            <div class="form-group">
                                <label for="nuevoEmail">Email</label>
                                <input type="email" id="nuevoEmail" class="form-control">
                            </div>
                            <div class="form-group">
                                <label for="nuevoTelefono">Teléfono</label>
                                <input type="tel" id="nuevoTelefono" class="form-control">
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label for="nuevoSelfie">Foto/Selfie *</label>
                            <div class="file-upload">
                                <input type="file" id="nuevoSelfie" accept="image/*" required>
                                <div class="file-upload-preview" id="previewContainer" style="display: none;">
                                    <img id="previewImage" alt="Preview">
                                    <button type="button" class="btn btn-sm btn-outline-danger" id="removePreview">
                                        <i class="fas fa-trash"></i>
                                    </button>
                                </div>
                            </div>
                            <div class="form-help">
                                Formatos: JPG, PNG. Tamaño máximo: 5MB
                            </div>
                        </div>
                        
                        <div class="form-group">
                            <label>
                                <input type="checkbox" id="procesarDescriptor" checked>
                                Generar descriptor facial automáticamente
                            </label>
                        </div>
                    </form>
                </div>
                
                <div class="modal-footer">
                    <button class="btn btn-secondary" id="btnCancelarNuevo">
                        Cancelar
                    </button>
                    <button class="btn btn-primary" id="btnCrearCliente">
                        <i class="fas fa-save"></i>
                        Crear Cliente
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Modal Importar -->
    <div class="modal-overlay" id="importarModal">
        <div class="modal-dialog modal-medium">
            <div class="modal-content">
                <button class="modal-close" id="closeImportModal">
                    <i class="fas fa-times"></i>
                </button>
                
                <div class="modal-header">
                    <h3>Importar Clientes</h3>
                </div>
                
                <div class="modal-body">
                    <div class="import-options">
                        <div class="import-method active" data-method="csv">
                            <i class="fas fa-file-csv"></i>
                            <h4>Archivo CSV</h4>
                            <p>Importar desde un archivo CSV con formato específico</p>
                        </div>
                        <div class="import-method" data-method="excel">
                            <i class="fas fa-file-excel"></i>
                            <h4>Archivo Excel</h4>
                            <p>Importar desde un archivo XLSX o XLS</p>
                        </div>
                    </div>
                    
                    <div class="file-upload-zone" id="importFileZone">
                        <input type="file" id="importFile" accept=".csv,.xlsx,.xls">
                        <div class="upload-content">
                            <i class="fas fa-cloud-upload-alt"></i>
                            <p>Arrastra tu archivo aquí o <strong>click para seleccionar</strong></p>
                            <small>Formatos soportados: CSV, XLSX, XLS</small>
                        </div>
                    </div>
                    
                    <div class="import-options-config" style="display: none;">
                        <h4>Opciones de Importación</h4>
                        <label>
                            <input type="checkbox" id="skipDuplicates" checked>
                            Omitir registros duplicados
                        </label>
                        <label>
                            <input type="checkbox" id="updateExisting">
                            Actualizar registros existentes
                        </label>
                    </div>
                    
                    <div class="import-preview" id="importPreview" style="display: none;">
                        <!-- Vista previa de los datos a importar -->
                    </div>
                </div>
                
                <div class="modal-footer">
                    <button class="btn btn-secondary" id="btnCancelarImport">
                        Cancelar
                    </button>
                    <button class="btn btn-primary" id="btnProcesarImport" disabled>
                        <i class="fas fa-upload"></i>
                        Importar Datos
                    </button>
                </div>
            </div>
        </div>
    </div>

    <!-- Loading Overlay -->
    <div class="loading-overlay" id="loadingOverlay">
        <div class="loading-spinner">
            <i class="fas fa-spinner fa-spin"></i>
            <span>Procesando...</span>
        </div>
    </div>

    <!-- Scripts -->
    <script src="assets/js/admin.js"></script>
    <script>
        // Gestión de Clientes
        class ClientesManager {
            constructor() {
                this.table = null;
                this.selectedClients = [];
                this.currentView = 'table';
                this.init();
            }

            async init() {
                this.initDataTable();
                this.setupEventListeners();
                await this.loadClientStats();
                await this.loadClients();
            }

            initDataTable() {
                this.table = $('#clientesTable').DataTable({
                    responsive: true,
                    pageLength: 25,
                    lengthMenu: [[10, 25, 50, 100], [10, 25, 50, 100]],
                    order: [[5, 'desc']], // Ordenar por fecha de registro
                    columnDefs: [
                        { orderable: false, targets: [0, 1, 8] }, // Checkbox, foto y acciones no ordenables
                        { searchable: false, targets: [0, 1, 8] },
                        { width: '50px', targets: [0, 1] },
                        { width: '100px', targets: 8 }
                    ],
                    language: {
                        url: 'https://cdn.datatables.net/plug-ins/1.13.7/i18n/es-ES.json'
                    },
                    dom: '<"table-controls"<"table-info"i><"table-pagination"p>>rt<"table-footer"<"table-length"l><"table-pagination"p>>',
                    drawCallback: () => {
                        this.updateSelection();
                    }
                });
            }

            setupEventListeners() {
                // Toggle selección completa
                document.getElementById('selectAll').addEventListener('change', (e) => {
                    this.toggleSelectAll(e.target.checked);
                });

                // Botones principales
                document.getElementById('btnNuevoCliente').addEventListener('click', () => {
                    this.showNuevoClienteModal();
                });

                document.getElementById('btnImportar').addEventListener('click', () => {
                    this.showImportModal();
                });

                document.getElementById('btnExportar').addEventListener('click', () => {
                    this.exportClients();
                });

                document.getElementById('btnLimpiarSeleccionados').addEventListener('click', () => {
                    this.deleteSelectedClients();
                });

                // Filtros
                document.getElementById('filtroFecha').addEventListener('change', (e) => {
                    this.applyFilters();
                });

                document.getElementById('filtroEstado').addEventListener('change', (e) => {
                    this.applyFilters();
                });

                document.getElementById('ordenarPor').addEventListener('change', (e) => {
                    this.applySorting(e.target.value);
                });

                document.getElementById('btnLimpiarFiltros').addEventListener('click', () => {
                    this.clearFilters();
                });

                // Vista toggle
                document.getElementById('vistaTabla').addEventListener('click', () => {
                    this.switchView('table');
                });

                document.getElementById('vistaTarjetas').addEventListener('click', () => {
                    this.switchView('cards');
                });

                // Refresh
                document.getElementById('refreshData').addEventListener('click', () => {
                    this.refreshData();
                });

                // Modal events
                this.setupModalEvents();
            }

            setupModalEvents() {
                // Cliente modal
                document.getElementById('closeClienteModal').addEventListener('click', () => {
                    document.getElementById('clienteModal').classList.remove('show');
                });

                document.getElementById('btnGuardarCliente').addEventListener('click', () => {
                    this.saveClienteChanges();
                });

                document.getElementById('btnEliminarCliente').addEventListener('click', () => {
                    this.deleteCurrentClient();
                });

                // Nuevo cliente modal
                document.getElementById('closeNuevoModal').addEventListener('click', () => {
                    document.getElementById('nuevoClienteModal').classList.remove('show');
                });

                document.getElementById('btnCrearCliente').addEventListener('click', () => {
                    this.createNewClient();
                });

                // Preview de imagen
                document.getElementById('nuevoSelfie').addEventListener('change', (e) => {
                    this.previewImage(e.target);
                });

                document.getElementById('removePreview').addEventListener('click', () => {
                    this.removeImagePreview();
                });
            }

            async loadClientStats() {
                try {
                    showLoading();
                    const response = await api.get('clientes/stats.php');
                    
                    if (response.success) {
                        document.getElementById('totalClientes').textContent = response.stats.total || 0;
                        document.getElementById('clientesHoy').textContent = response.stats.hoy || 0;
                        document.getElementById('clientesActivos').textContent = response.stats.activos || 0;
                        document.getElementById('clientesDuplicados').textContent = response.stats.duplicados || 0;
                        
                        // Actualizar badge del sidebar
                        document.getElementById('clientesBadge').textContent = response.stats.total || 0;
                    }
                } catch (error) {
                    showError('Error cargando estadísticas');
                } finally {
                    hideLoading();
                }
            }

            async loadClients() {
                try {
                    showLoading();
                    const response = await api.get('clientes/list.php');
                    
                    if (response.success) {
                        this.renderClients(response.data);
                    } else {
                        throw new Error(response.message);
                    }
                } catch (error) {
                    showError('Error cargando clientes: ' + error.message);
                } finally {
                    hideLoading();
                }
            }

            renderClients(clients) {
                // Limpiar tabla
                this.table.clear();

                clients.forEach(client => {
                    const row = [
                        `<input type="checkbox" class="client-checkbox" data-id="${client.id}">`,
                        this.renderClientPhoto(client),
                        this.renderClientName(client),
                        client.email || '-',
                        client.telefono || '-',
                        Utils.formatDate(client.creado_en),
                        this.renderLastValidation(client),
                        this.renderClientStatus(client),
                        this.renderClientActions(client)
                    ];

                    this.table.row.add(row);
                });

                this.table.draw();
            }

            renderClientPhoto(client) {
                if (client.selfie_path) {
                    return `
                        <div class="client-photo-thumb">
                            <img src="../${client.selfie_path}" alt="${client.nombre}" 
                                 onclick="clientesManager.showClientModal(${client.id})">
                        </div>
                    `;
                }
                return `
                    <div class="client-photo-thumb no-photo">
                        <i class="fas fa-user"></i>
                    </div>
                `;
            }

            renderClientName(client) {
                return `
                    <div class="client-name-cell">
                        <strong>${client.nombre}</strong>
                        <small>ID: ${client.id}</small>
                    </div>
                `;
            }

            renderLastValidation(client) {
                if (client.ultima_validacion) {
                    return `
                        <span class="validation-date success">
                            ${Utils.formatTime(client.ultima_validacion)}
                        </span>
                    `;
                }
                return `<span class="validation-date never">Nunca</span>`;
            }

            renderClientStatus(client) {
                const status = client.estado || 'activo';
                const statusClass = {
                    'activo': 'success',
                    'inactivo': 'warning', 
                    'bloqueado': 'danger'
                };

                return `
                    <span class="status-badge ${statusClass[status]}">
                        ${status.charAt(0).toUpperCase() + status.slice(1)}
                    </span>
                `;
            }

            renderClientActions(client) {
                return `
                    <div class="action-buttons">
                        <button class="btn btn-sm btn-outline-primary" 
                                onclick="clientesManager.showClientModal(${client.id})"
                                title="Ver detalles">
                            <i class="fas fa-eye"></i>
                        </button>
                        <button class="btn btn-sm btn-outline-success" 
                                onclick="clientesManager.editClient(${client.id})"
                                title="Editar">
                            <i class="fas fa-edit"></i>
                        </button>
                        <button class="btn btn-sm btn-outline-danger" 
                                onclick="clientesManager.deleteClient(${client.id})"
                                title="Eliminar">
                            <i class="fas fa-trash"></i>
                        </button>
                    </div>
                `;
            }

            async showClientModal(clientId) {
                try {
                    showLoading();
                    const response = await api.get(`clientes/details.php?id=${clientId}`);
                    
                    if (response.success) {
                        this.populateClientModal(response.data);
                        document.getElementById('clienteModal').classList.add('show');
                    } else {
                        throw new Error(response.message);
                    }
                } catch (error) {
                    showError('Error cargando detalles del cliente');
                } finally {
                    hideLoading();
                }
            }

            populateClientModal(client) {
                document.getElementById('clienteId').value = client.id;
                document.getElementById('clienteNombre').value = client.nombre;
                document.getElementById('clienteEmail').value = client.email || '';
                document.getElementById('clienteTelefono').value = client.telefono || '';
                document.getElementById('clienteRegistro').value = Utils.formatDate(client.creado_en);
                document.getElementById('clienteEstado').value = client.estado || 'activo';

                // Foto
                const fotoImg = document.getElementById('clienteFoto');
                const placeholder = document.getElementById('photoPlaceholder');
                
                if (client.selfie_path) {
                    fotoImg.src = '../' + client.selfie_path;
                    fotoImg.style.display = 'block';
                    placeholder.style.display = 'none';
                    document.getElementById('photoDate').textContent = 
                        'Registrada: ' + Utils.formatDate(client.creado_en);
                } else {
                    fotoImg.style.display = 'none';
                    placeholder.style.display = 'flex';
                }

                // Historial (simulado por ahora)
                this.loadClientHistory(client.id);
            }

            async loadClientHistory(clientId) {
                try {
                    const response = await api.get(`clientes/history.php?id=${clientId}`);
                    
                    if (response.success) {
                        const container = document.getElementById('historialValidaciones');
                        
                        if (response.data.length === 0) {
                            container.innerHTML = `
                                <div class="no-history">
                                    <i class="fas fa-info-circle"></i>
                                    No hay validaciones registradas para este cliente
                                </div>
                            `;
                            return;
                        }

                        container.innerHTML = response.data.map(validation => `
                            <div class="history-item ${validation.resultado}">
                                <div class="history-icon">
                                    <i class="fas ${validation.resultado === 'exitoso' ? 'fa-check' : 'fa-times'}"></i>
                                </div>
                                <div class="history-content">
                                    <div class="history-text">
                                        ${validation.resultado === 'exitoso' ? 'Validación exitosa' : 'Validación fallida'}
                                        ${validation.confianza ? `(${Math.round(validation.confianza * 100)}% confianza)` : ''}
                                    </div>
                                    <div class="history-time">${Utils.formatTime(validation.timestamp)}</div>
                                </div>
                            </div>
                        `).join('');
                    }
                } catch (error) {
                    console.error('Error loading history:', error);
                }
            }

            showNuevoClienteModal() {
                document.getElementById('nuevoClienteForm').reset();
                this.removeImagePreview();
                document.getElementById('nuevoClienteModal').classList.add('show');
            }

            previewImage(input) {
                const file = input.files[0];
                if (file) {
                    const reader = new FileReader();
                    reader.onload = (e) => {
                        document.getElementById('previewImage').src = e.target.result;
                        document.getElementById('previewContainer').style.display = 'block';
                    };
                    reader.readAsDataURL(file);
                }
            }

            removeImagePreview() {
                document.getElementById('previewContainer').style.display = 'none';
                document.getElementById('nuevoSelfie').value = '';
            }

            async createNewClient() {
                const form = document.getElementById('nuevoClienteForm');
                const formData = new FormData(form);
                
                try {
                    showLoading('Creando cliente...');
                    const response = await api.uploadFile('clientes/create.php', 
                        document.getElementById('nuevoSelfie').files[0], {
                            nombre: document.getElementById('nuevoNombre').value,
                            email: document.getElementById('nuevoEmail').value,
                            telefono: document.getElementById('nuevoTelefono').value,
                            procesarDescriptor: document.getElementById('procesarDescriptor').checked
                        }
                    );
                    
                    if (response.success) {
                        showSuccess('Cliente creado exitosamente');
                        document.getElementById('nuevoClienteModal').classList.remove('show');
                        this.refreshData();
                    } else {
                        throw new Error(response.message);
                    }
                } catch (error) {
                    showError('Error creando cliente: ' + error.message);
                } finally {
                    hideLoading();
                }
            }

            async refreshData() {
                await this.loadClientStats();
                await this.loadClients();
                showSuccess('Datos actualizados');
            }

            switchView(view) {
                this.currentView = view;
                
                const tableView = document.querySelector('.table-container');
                const cardsView = document.getElementById('clientesCards');
                const tableBtn = document.getElementById('vistaTabla');
                const cardsBtn = document.getElementById('vistaTarjetas');

                if (view === 'table') {
                    tableView.style.display = 'block';
                    cardsView.style.display = 'none';
                    tableBtn.classList.add('active');
                    cardsBtn.classList.remove('active');
                } else {
                    tableView.style.display = 'none';
                    cardsView.style.display = 'grid';
                    cardsBtn.classList.add('active');
                    tableBtn.classList.remove('active');
                    this.renderCardsView();
                }
            }

            renderCardsView() {
                // Implementar vista de tarjetas
                console.log('Rendering cards view...');
            }

            toggleSelectAll(checked) {
                const checkboxes = document.querySelectorAll('.client-checkbox');
                checkboxes.forEach(cb => {
                    cb.checked = checked;
                });
                this.updateSelection();
            }

            updateSelection() {
                const checkboxes = document.querySelectorAll('.client-checkbox:checked');
                this.selectedClients = Array.from(checkboxes).map(cb => cb.dataset.id);
                
                const deleteBtn = document.getElementById('btnLimpiarSeleccionados');
                deleteBtn.disabled = this.selectedClients.length === 0;
                deleteBtn.innerHTML = `
                    <i class="fas fa-trash"></i>
                    Eliminar Selec. (${this.selectedClients.length})
                `;
            }

            async exportClients() {
                try {
                    showLoading('Generando exportación...');
                    
                    const response = await fetch('api/clientes/export.php');
                    const blob = await response.blob();
                    
                    const url = window.URL.createObjectURL(blob);
                    const a = document.createElement('a');
                    a.href = url;
                    a.download = `clientes_${new Date().toISOString().split('T')[0]}.csv`;
                    a.click();
                    
                    showSuccess('Exportación completada');
                    
                } catch (error) {
                    showError('Error en la exportación');
                } finally {
                    hideLoading();
                }
            }

            applyFilters() {
                // Implementar filtros
                console.log('Applying filters...');
            }

            clearFilters() {
                document.getElementById('filtroFecha').value = '';
                document.getElementById('filtroEstado').value = '';
                document.getElementById('ordenarPor').value = 'creado_desc';
                this.applyFilters();
            }

            applySorting(sortBy) {
                // Implementar ordenamiento
                console.log('Applying sort:', sortBy);
            }
        }

        // Inicializar cuando el DOM esté listo
        document.addEventListener('DOMContentLoaded', () => {
            window.clientesManager = new ClientesManager();
        });
    </script>
</body>
</html>