/* =====================================================
   JAVASCRIPT PRINCIPAL - PANEL ADMINISTRACIÓN
   ===================================================== */

// Variables globales
let currentUser = null;
let sidebarCollapsed = false;

// Utilidades para mostrar/ocultar loading
function showLoading(message = 'Cargando...') {
    const overlay = document.getElementById('loadingOverlay');
    if (overlay) {
        const span = overlay.querySelector('span');
        if (span) span.textContent = message;
        overlay.classList.add('show');
    }
}

function hideLoading() {
    const overlay = document.getElementById('loadingOverlay');
    if (overlay) {
        overlay.classList.remove('show');
    }
}

// Sistema de notificaciones
function showNotification(message, type = 'info', duration = 5000) {
    // Crear elemento de notificación si no existe
    let container = document.getElementById('notificationContainer');
    if (!container) {
        container = document.createElement('div');
        container.id = 'notificationContainer';
        container.className = 'notification-container';
        document.body.appendChild(container);
    }

    const notification = document.createElement('div');
    notification.className = `notification notification-${type}`;
    
    // Iconos según el tipo
    const icons = {
        success: 'fa-check-circle',
        error: 'fa-exclamation-circle',
        warning: 'fa-exclamation-triangle',
        info: 'fa-info-circle'
    };
    
    notification.innerHTML = `
        <div class="notification-content">
            <i class="fas ${icons[type]}"></i>
            <span>${message}</span>
        </div>
        <button class="notification-close">
            <i class="fas fa-times"></i>
        </button>
    `;

    container.appendChild(notification);

    // Mostrar notificación
    setTimeout(() => notification.classList.add('show'), 100);

    // Auto-ocultar
    const autoHide = setTimeout(() => {
        hideNotification(notification);
    }, duration);

    // Cerrar manualmente
    notification.querySelector('.notification-close').addEventListener('click', () => {
        clearTimeout(autoHide);
        hideNotification(notification);
    });

    return notification;
}

function hideNotification(notification) {
    notification.classList.remove('show');
    setTimeout(() => {
        if (notification.parentNode) {
            notification.parentNode.removeChild(notification);
        }
    }, 300);
}

// Funciones de conveniencia
function showSuccess(message) {
    return showNotification(message, 'success');
}

function showError(message) {
    return showNotification(message, 'error');
}

function showWarning(message) {
    return showNotification(message, 'warning');
}

function showInfo(message) {
    return showNotification(message, 'info');
}

// Sidebar Management
class SidebarManager {
    constructor() {
        this.sidebar = document.getElementById('sidebar');
        this.mainContent = document.getElementById('mainContent');
        this.mobileToggle = document.getElementById('mobileToggle');
        this.sidebarToggle = document.getElementById('sidebarToggle');
        
        this.init();
    }

    init() {
        // Toggle móvil
        if (this.mobileToggle) {
            this.mobileToggle.addEventListener('click', () => {
                this.toggleMobile();
            });
        }

        // Toggle desktop
        if (this.sidebarToggle) {
            this.sidebarToggle.addEventListener('click', () => {
                this.toggleDesktop();
            });
        }

        // Cerrar sidebar en móvil cuando se hace click fuera
        document.addEventListener('click', (e) => {
            if (window.innerWidth <= 768 && 
                !this.sidebar.contains(e.target) && 
                !this.mobileToggle.contains(e.target) &&
                this.sidebar.classList.contains('show')) {
                this.closeMobile();
            }
        });

        // Responsive handling
        window.addEventListener('resize', () => {
            this.handleResize();
        });

        // Marcar item activo
        this.setActiveNavItem();
    }

    toggleMobile() {
        this.sidebar.classList.toggle('show');
    }

    closeMobile() {
        this.sidebar.classList.remove('show');
    }

    toggleDesktop() {
        sidebarCollapsed = !sidebarCollapsed;
        document.body.classList.toggle('sidebar-collapsed', sidebarCollapsed);
        
        // Guardar preferencia
        localStorage.setItem('sidebar-collapsed', sidebarCollapsed);
    }

    handleResize() {
        if (window.innerWidth > 768) {
            this.sidebar.classList.remove('show');
        }
    }

    setActiveNavItem() {
        const currentPage = window.location.pathname.split('/').pop() || 'index.php';
        const navLinks = document.querySelectorAll('.nav-link');
        
        navLinks.forEach(link => {
            const href = link.getAttribute('href');
            const parentItem = link.closest('.nav-item');
            
            if (href === currentPage || (currentPage === '' && href === 'index.php')) {
                parentItem.classList.add('active');
            } else {
                parentItem.classList.remove('active');
            }
        });
    }
}

// API Helper Class
class ApiClient {
    constructor(baseUrl = 'api/') {
        this.baseUrl = baseUrl;
        this.defaultHeaders = {
            'Content-Type': 'application/json',
        };
    }

    async request(endpoint, options = {}) {
        const url = this.baseUrl + endpoint;
        const config = {
            headers: { ...this.defaultHeaders, ...options.headers },
            ...options
        };

        try {
            const response = await fetch(url, config);
            
            if (!response.ok) {
                throw new Error(`HTTP error! status: ${response.status}`);
            }

            const contentType = response.headers.get('content-type');
            if (contentType && contentType.includes('application/json')) {
                return await response.json();
            } else {
                return await response.text();
            }
        } catch (error) {
            console.error('API request failed:', error);
            throw error;
        }
    }

    async get(endpoint, params = {}) {
        const queryString = new URLSearchParams(params).toString();
        const url = queryString ? `${endpoint}?${queryString}` : endpoint;
        
        return this.request(url, {
            method: 'GET'
        });
    }

    async post(endpoint, data = {}) {
        return this.request(endpoint, {
            method: 'POST',
            body: JSON.stringify(data)
        });
    }

    async put(endpoint, data = {}) {
        return this.request(endpoint, {
            method: 'PUT',
            body: JSON.stringify(data)
        });
    }

    async delete(endpoint) {
        return this.request(endpoint, {
            method: 'DELETE'
        });
    }

    // Métodos específicos para uploads
    async uploadFile(endpoint, file, additionalData = {}) {
        const formData = new FormData();
        formData.append('file', file);
        
        Object.keys(additionalData).forEach(key => {
            formData.append(key, additionalData[key]);
        });

        return this.request(endpoint, {
            method: 'POST',
            headers: {}, // Dejar que el browser establezca Content-Type para FormData
            body: formData
        });
    }
}

// Inicializar cliente API global
const api = new ApiClient();

// Modal Manager
class ModalManager {
    constructor() {
        this.activeModals = [];
        this.init();
    }

    init() {
        // Cerrar modal con ESC
        document.addEventListener('keydown', (e) => {
            if (e.key === 'Escape' && this.activeModals.length > 0) {
                this.close(this.activeModals[this.activeModals.length - 1]);
            }
        });
    }

    create(content, options = {}) {
        const modal = document.createElement('div');
        modal.className = 'modal-overlay';
        
        const defaultOptions = {
            closable: true,
            size: 'medium',
            backdrop: true
        };

        const config = { ...defaultOptions, ...options };

        modal.innerHTML = `
            <div class="modal-dialog modal-${config.size}">
                <div class="modal-content">
                    ${config.closable ? '<button class="modal-close"><i class="fas fa-times"></i></button>' : ''}
                    ${content}
                </div>
            </div>
        `;

        document.body.appendChild(modal);

        // Event listeners
        if (config.closable) {
            const closeBtn = modal.querySelector('.modal-close');
            closeBtn.addEventListener('click', () => this.close(modal));
        }

        if (config.backdrop) {
            modal.addEventListener('click', (e) => {
                if (e.target === modal) {
                    this.close(modal);
                }
            });
        }

        return modal;
    }

    show(modal) {
        this.activeModals.push(modal);
        document.body.classList.add('modal-open');
        
        setTimeout(() => {
            modal.classList.add('show');
        }, 10);

        return modal;
    }

    close(modal) {
        modal.classList.remove('show');
        
        setTimeout(() => {
            if (modal.parentNode) {
                modal.parentNode.removeChild(modal);
            }
            
            const index = this.activeModals.indexOf(modal);
            if (index > -1) {
                this.activeModals.splice(index, 1);
            }

            if (this.activeModals.length === 0) {
                document.body.classList.remove('modal-open');
            }
        }, 300);
    }

    confirm(message, title = 'Confirmar') {
        return new Promise((resolve) => {
            const content = `
                <div class="modal-header">
                    <h3>${title}</h3>
                </div>
                <div class="modal-body">
                    <p>${message}</p>
                </div>
                <div class="modal-footer">
                    <button class="btn btn-secondary" data-action="cancel">Cancelar</button>
                    <button class="btn btn-primary" data-action="confirm">Confirmar</button>
                </div>
            `;

            const modal = this.create(content, { size: 'small' });
            
            modal.addEventListener('click', (e) => {
                const action = e.target.getAttribute('data-action');
                if (action === 'confirm') {
                    this.close(modal);
                    resolve(true);
                } else if (action === 'cancel') {
                    this.close(modal);
                    resolve(false);
                }
            });

            this.show(modal);
        });
    }

    alert(message, title = 'Información') {
        return new Promise((resolve) => {
            const content = `
                <div class="modal-header">
                    <h3>${title}</h3>
                </div>
                <div class="modal-body">
                    <p>${message}</p>
                </div>
                <div class="modal-footer">
                    <button class="btn btn-primary" data-action="ok">Aceptar</button>
                </div>
            `;

            const modal = this.create(content, { size: 'small' });
            
            modal.addEventListener('click', (e) => {
                const action = e.target.getAttribute('data-action');
                if (action === 'ok') {
                    this.close(modal);
                    resolve();
                }
            });

            this.show(modal);
        });
    }
}

// Instanciar modal manager global
const modals = new ModalManager();

// Utilidades de formateo
const Utils = {
    formatDate(date, includeTime = true) {
        const d = new Date(date);
        const options = {
            year: 'numeric',
            month: 'short',
            day: 'numeric'
        };
        
        if (includeTime) {
            options.hour = '2-digit';
            options.minute = '2-digit';
        }
        
        return d.toLocaleDateString('es-ES', options);
    },

    formatTime(timestamp) {
        const date = new Date(timestamp);
        const now = new Date();
        const diff = now - date;
        
        if (diff < 60000) return 'Hace un momento';
        if (diff < 3600000) return `Hace ${Math.floor(diff/60000)} min`;
        if (diff < 86400000) return `Hace ${Math.floor(diff/3600000)} h`;
        if (diff < 2592000000) return `Hace ${Math.floor(diff/86400000)} días`;
        
        return date.toLocaleDateString('es-ES');
    },

    formatFileSize(bytes) {
        const sizes = ['Bytes', 'KB', 'MB', 'GB'];
        if (bytes === 0) return '0 Bytes';
        const i = Math.floor(Math.log(bytes) / Math.log(1024));
        return Math.round(bytes / Math.pow(1024, i) * 100) / 100 + ' ' + sizes[i];
    },

    formatNumber(num) {
        return new Intl.NumberFormat('es-ES').format(num);
    },

    validateEmail(email) {
        const re = /^[^\s@]+@[^\s@]+\.[^\s@]+$/;
        return re.test(email);
    },

    validatePhone(phone) {
        const re = /^[\+]?[0-9\s\-\(\)]+$/;
        return re.test(phone) && phone.length >= 8;
    },

    sanitizeInput(str) {
        const div = document.createElement('div');
        div.textContent = str;
        return div.innerHTML;
    },

    generateId() {
        return Date.now().toString(36) + Math.random().toString(36).substr(2);
    },

    debounce(func, wait, immediate) {
        let timeout;
        return function executedFunction(...args) {
            const later = () => {
                timeout = null;
                if (!immediate) func(...args);
            };
            const callNow = immediate && !timeout;
            clearTimeout(timeout);
            timeout = setTimeout(later, wait);
            if (callNow) func(...args);
        };
    },

    throttle(func, limit) {
        let inThrottle;
        return function() {
            const args = arguments;
            const context = this;
            if (!inThrottle) {
                func.apply(context, args);
                inThrottle = true;
                setTimeout(() => inThrottle = false, limit);
            }
        }
    }
};

// Session Manager
class SessionManager {
    constructor() {
        this.checkInterval = null;
        this.warningShown = false;
        this.init();
    }

    init() {
        this.startSessionCheck();
        this.setupActivityTracking();
    }

    startSessionCheck() {
        // Verificar sesión cada 5 minutos
        this.checkInterval = setInterval(() => {
            this.checkSession();
        }, 300000);
    }

    async checkSession() {
        try {
            const response = await api.get('auth/check.php');
            
            if (!response.success) {
                this.handleSessionExpired();
            } else if (response.warning && !this.warningShown) {
                this.showExpirationWarning(response.remaining);
            }
        } catch (error) {
            console.error('Session check failed:', error);
        }
    }

    handleSessionExpired() {
        if (this.checkInterval) {
            clearInterval(this.checkInterval);
        }
        
        modals.alert(
            'Tu sesión ha expirado. Serás redirigido al login.',
            'Sesión Expirada'
        ).then(() => {
            window.location.href = 'login.php';
        });
    }

    showExpirationWarning(remaining) {
        this.warningShown = true;
        
        showWarning(
            `Tu sesión expirará en ${remaining} minutos. Guarda tu trabajo.`
        );
        
        // Reset warning después de 10 minutos
        setTimeout(() => {
            this.warningShown = false;
        }, 600000);
    }

    setupActivityTracking() {
        let lastActivity = Date.now();
        
        const events = ['mousedown', 'mousemove', 'keypress', 'scroll', 'touchstart'];
        
        const updateActivity = Utils.throttle(() => {
            lastActivity = Date.now();
        }, 60000); // Una vez por minuto máximo

        events.forEach(event => {
            document.addEventListener(event, updateActivity, true);
        });
    }

    async extendSession() {
        try {
            const response = await api.post('auth/extend.php');
            
            if (response.success) {
                showSuccess('Sesión extendida exitosamente');
                this.warningShown = false;
            }
        } catch (error) {
            showError('No se pudo extender la sesión');
        }
    }
}

// Data Table Helper
class DataTableHelper {
    constructor(tableId, options = {}) {
        this.table = document.getElementById(tableId);
        this.options = {
            searchable: true,
            sortable: true,
            paginated: true,
            pageSize: 10,
            ...options
        };
        this.data = [];
        this.filteredData = [];
        this.currentPage = 1;
        this.sortColumn = null;
        this.sortDirection = 'asc';
        this.searchTerm = '';
        
        this.init();
    }

    init() {
        if (this.options.searchable) {
            this.createSearchBox();
        }
        
        if (this.options.sortable) {
            this.enableSorting();
        }
        
        if (this.options.paginated) {
            this.createPagination();
        }
    }

    setData(data) {
        this.data = data;
        this.filteredData = [...data];
        this.render();
    }

    createSearchBox() {
        const container = this.table.parentNode;
        const searchBox = document.createElement('div');
        searchBox.className = 'datatable-search';
        searchBox.innerHTML = `
            <input type="text" placeholder="Buscar..." class="search-input">
        `;
        
        container.insertBefore(searchBox, this.table);
        
        const input = searchBox.querySelector('.search-input');
        input.addEventListener('input', Utils.debounce((e) => {
            this.search(e.target.value);
        }, 300));
    }

    search(term) {
        this.searchTerm = term.toLowerCase();
        this.filteredData = this.data.filter(row => {
            return Object.values(row).some(value => 
                String(value).toLowerCase().includes(this.searchTerm)
            );
        });
        this.currentPage = 1;
        this.render();
    }

    enableSorting() {
        this.table.addEventListener('click', (e) => {
            const th = e.target.closest('th');
            if (th && th.dataset.sortable !== 'false') {
                const column = th.dataset.column;
                this.sort(column);
            }
        });
    }

    sort(column) {
        if (this.sortColumn === column) {
            this.sortDirection = this.sortDirection === 'asc' ? 'desc' : 'asc';
        } else {
            this.sortColumn = column;
            this.sortDirection = 'asc';
        }

        this.filteredData.sort((a, b) => {
            let aVal = a[column];
            let bVal = b[column];
            
            // Detectar tipo de dato
            if (!isNaN(aVal) && !isNaN(bVal)) {
                aVal = Number(aVal);
                bVal = Number(bVal);
            }
            
            if (aVal < bVal) return this.sortDirection === 'asc' ? -1 : 1;
            if (aVal > bVal) return this.sortDirection === 'asc' ? 1 : -1;
            return 0;
        });

        this.updateSortHeaders();
        this.render();
    }

    updateSortHeaders() {
        const headers = this.table.querySelectorAll('th');
        headers.forEach(th => {
            th.classList.remove('sort-asc', 'sort-desc');
            if (th.dataset.column === this.sortColumn) {
                th.classList.add(`sort-${this.sortDirection}`);
            }
        });
    }

    createPagination() {
        const container = this.table.parentNode;
        const pagination = document.createElement('div');
        pagination.className = 'datatable-pagination';
        container.appendChild(pagination);
    }

    render() {
        this.renderTable();
        if (this.options.paginated) {
            this.renderPagination();
        }
    }

    renderTable() {
        // Implementar rendering de tabla
        // Esta función debe ser sobrescrita por cada implementación específica
        console.log('DataTable render - override this method');
    }

    renderPagination() {
        const totalItems = this.filteredData.length;
        const totalPages = Math.ceil(totalItems / this.options.pageSize);
        const pagination = this.table.parentNode.querySelector('.datatable-pagination');
        
        const startIndex = (this.currentPage - 1) * this.options.pageSize;
        const endIndex = Math.min(startIndex + this.options.pageSize, totalItems);
        
        pagination.innerHTML = `
            <div class="pagination-info">
                Mostrando ${startIndex + 1}-${endIndex} de ${totalItems} registros
            </div>
            <div class="pagination-controls">
                <button ${this.currentPage <= 1 ? 'disabled' : ''} data-page="${this.currentPage - 1}">
                    <i class="fas fa-chevron-left"></i>
                </button>
                <span class="page-numbers"></span>
                <button ${this.currentPage >= totalPages ? 'disabled' : ''} data-page="${this.currentPage + 1}">
                    <i class="fas fa-chevron-right"></i>
                </button>
            </div>
        `;

        // Event listeners para paginación
        pagination.addEventListener('click', (e) => {
            const button = e.target.closest('button[data-page]');
            if (button && !button.disabled) {
                this.goToPage(parseInt(button.dataset.page));
            }
        });
    }

    goToPage(page) {
        const totalPages = Math.ceil(this.filteredData.length / this.options.pageSize);
        if (page >= 1 && page <= totalPages) {
            this.currentPage = page;
            this.render();
        }
    }

    getCurrentPageData() {
        const startIndex = (this.currentPage - 1) * this.options.pageSize;
        return this.filteredData.slice(startIndex, startIndex + this.options.pageSize);
    }
}

// Inicialización global
document.addEventListener('DOMContentLoaded', () => {
    // Inicializar componentes principales
    window.sidebarManager = new SidebarManager();
    window.sessionManager = new SessionManager();
    
    // Recuperar estado del sidebar
    const sidebarState = localStorage.getItem('sidebar-collapsed');
    if (sidebarState === 'true') {
        sidebarCollapsed = true;
        document.body.classList.add('sidebar-collapsed');
    }
    
    // Configurar CSRF token si existe
    const csrfToken = document.querySelector('meta[name="csrf-token"]');
    if (csrfToken) {
        api.defaultHeaders['X-CSRF-TOKEN'] = csrfToken.getAttribute('content');
    }
    
    // Log para debug
    console.log('Admin panel initialized successfully');
});

// Estilos adicionales para componentes JS (agregar al CSS)
const additionalCSS = `
/* Notifications */
.notification-container {
    position: fixed;
    top: 20px;
    right: 20px;
    z-index: 10000;
    max-width: 400px;
}

.notification {
    background: white;
    border-radius: 8px;
    box-shadow: 0 10px 25px rgba(0,0,0,0.15);
    margin-bottom: 10px;
    padding: 16px;
    display: flex;
    align-items: center;
    justify-content: space-between;
    transform: translateX(100%);
    opacity: 0;
    transition: all 0.3s ease;
    border-left: 4px solid #3b82f6;
}

.notification.show {
    transform: translateX(0);
    opacity: 1;
}

.notification-success { border-left-color: #10b981; }
.notification-error { border-left-color: #ef4444; }
.notification-warning { border-left-color: #f59e0b; }

.notification-content {
    display: flex;
    align-items: center;
    gap: 12px;
    flex: 1;
}

.notification-close {
    background: none;
    border: none;
    color: #6b7280;
    cursor: pointer;
    padding: 4px;
    border-radius: 4px;
    transition: background 0.15s ease;
}

.notification-close:hover {
    background: #f3f4f6;
}

/* Modals */
.modal-overlay {
    position: fixed;
    top: 0;
    left: 0;
    right: 0;
    bottom: 0;
    background: rgba(0,0,0,0.5);
    display: flex;
    align-items: center;
    justify-content: center;
    z-index: 9999;
    opacity: 0;
    visibility: hidden;
    transition: all 0.3s ease;
}

.modal-overlay.show {
    opacity: 1;
    visibility: visible;
}

.modal-dialog {
    background: white;
    border-radius: 12px;
    max-height: 90vh;
    overflow-y: auto;
    box-shadow: 0 20px 40px rgba(0,0,0,0.15);
    transform: scale(0.9) translateY(-50px);
    transition: all 0.3s ease;
    position: relative;
}

.modal-overlay.show .modal-dialog {
    transform: scale(1) translateY(0);
}

.modal-small { max-width: 400px; width: 90%; }
.modal-medium { max-width: 600px; width: 90%; }
.modal-large { max-width: 900px; width: 95%; }

.modal-content {
    position: relative;
}

.modal-close {
    position: absolute;
    top: 16px;
    right: 16px;
    background: none;
    border: none;
    font-size: 16px;
    color: #6b7280;
    cursor: pointer;
    width: 32px;
    height: 32px;
    border-radius: 50%;
    display: flex;
    align-items: center;
    justify-content: center;
    transition: all 0.15s ease;
    z-index: 1;
}

.modal-close:hover {
    background: #f3f4f6;
    color: #374151;
}

.modal-header {
    padding: 24px 24px 16px;
    border-bottom: 1px solid #e5e7eb;
}

.modal-header h3 {
    margin: 0;
    font-size: 1.25rem;
    font-weight: 600;
    color: #1f2937;
}

.modal-body {
    padding: 24px;
}

.modal-footer {
    padding: 16px 24px 24px;
    display: flex;
    gap: 12px;
    justify-content: flex-end;
    border-top: 1px solid #e5e7eb;
}

body.modal-open {
    overflow: hidden;
}
`;

// Inyectar estilos adicionales
const style = document.createElement('style');
style.textContent = additionalCSS;
document.head.appendChild(style);