<?php
// /admin/api/dashboard.php
session_start();
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') {
    exit;
}

// Verificar autenticación
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'No autorizado']);
    exit;
}

require_once __DIR__ . '/../../bd/bd.php';

class DashboardAPI {
    private $pdo;
    private $admin_id;

    public function __construct($pdo) {
        $this->pdo = $pdo;
        $this->admin_id = $_SESSION['admin_id'] ?? null;
        
        // Log de acceso al dashboard
        $this->logActivity('ver_dashboard', null, null, ['seccion' => 'estadisticas']);
    }

    public function getDashboardData() {
        try {
            $period = isset($_GET['period']) ? (int)$_GET['period'] : 30;
            
            $data = [
                'success' => true,
                'stats' => $this->getStats(),
                'charts' => $this->getChartData($period),
                'activity' => $this->getRecentActivity(),
                'system' => $this->getSystemStatus()
            ];
            
            return $data;
            
        } catch (Exception $e) {
            return [
                'success' => false,
                'message' => 'Error obteniendo datos del dashboard',
                'debug' => $e->getMessage()
            ];
        }
    }

    private function getStats() {
        // Estadísticas básicas usando la vista creada
        $stmt = $this->pdo->query("SELECT * FROM dashboard_stats");
        $basicStats = $stmt->fetch(PDO::FETCH_ASSOC);

        // Estadísticas adicionales
        $additionalStats = [];

        // Registros de la semana pasada para comparar crecimiento
        $stmt = $this->pdo->prepare("
            SELECT COUNT(*) as registros_semana_pasada 
            FROM users 
            WHERE creado_en >= DATE_SUB(NOW(), INTERVAL 14 DAY) 
            AND creado_en < DATE_SUB(NOW(), INTERVAL 7 DAY)
        ");
        $stmt->execute();
        $weeklyComparison = $stmt->fetch(PDO::FETCH_ASSOC);

        // Calcular porcentaje de crecimiento
        $registrosEstaSemanaa = (int)$basicStats['registros_semana'];
        $registrosSemanaAnterior = (int)$weeklyComparison['registros_semana_pasada'];
        $crecimiento = 0;

        if ($registrosSemanaAnterior > 0) {
            $crecimiento = (($registrosEstaSemanaa - $registrosSemanaAnterior) / $registrosSemanaAnterior) * 100;
        } elseif ($registrosEstaSemanaa > 0) {
            $crecimiento = 100;
        }

        // Promedio de validaciones por día (últimos 7 días)
        $stmt = $this->pdo->query("
            SELECT AVG(daily_validations) as promedio_validaciones_dia
            FROM (
                SELECT DATE(timestamp) as fecha, COUNT(*) as daily_validations
                FROM validation_logs 
                WHERE timestamp >= DATE_SUB(NOW(), INTERVAL 7 DAY)
                GROUP BY DATE(timestamp)
            ) as daily_stats
        ");
        $avgValidations = $stmt->fetch(PDO::FETCH_ASSOC);

        // Hora pico (hora con más validaciones exitosas)
        $stmt = $this->pdo->query("
            SELECT HOUR(timestamp) as hora, COUNT(*) as total
            FROM validation_logs 
            WHERE resultado = 'exitoso' AND timestamp >= DATE_SUB(NOW(), INTERVAL 30 DAY)
            GROUP BY HOUR(timestamp)
            ORDER BY total DESC
            LIMIT 1
        ");
        $horaPico = $stmt->fetch(PDO::FETCH_ASSOC);

        return array_merge($basicStats ?: [], [
            'crecimiento_semanal' => round($crecimiento, 1),
            'promedio_validaciones_dia' => round($avgValidations['promedio_validaciones_dia'] ?? 0, 1),
            'hora_pico' => $horaPico ? $horaPico['hora'] . ':00' : 'N/A',
            'total_validaciones_mes' => $this->getValidacionesMes()
        ]);
    }

    private function getValidacionesMes() {
        $stmt = $this->pdo->query("
            SELECT COUNT(*) as total
            FROM validation_logs 
            WHERE timestamp >= DATE_SUB(NOW(), INTERVAL 30 DAY)
        ");
        $result = $stmt->fetch(PDO::FETCH_ASSOC);
        return $result['total'] ?? 0;
    }

    private function getChartData($period = 30) {
        $charts = [];

        // Gráfico de registros por día
        $charts['registros'] = $this->getRegistrosChart($period);
        
        // Gráfico de validaciones por estado
        $charts['validaciones'] = $this->getValidacionesChart();
        
        // Gráfico adicional: Registros por hora del día
        $charts['horarios'] = $this->getHorarios();
        
        return $charts;
    }

    private function getRegistrosChart($period) {
        $stmt = $this->pdo->prepare("
            WITH RECURSIVE date_series AS (
                SELECT DATE_SUB(CURDATE(), INTERVAL ? DAY) as fecha
                UNION ALL
                SELECT DATE_ADD(fecha, INTERVAL 1 DAY)
                FROM date_series 
                WHERE fecha < CURDATE()
            )
            SELECT 
                ds.fecha,
                COALESCE(COUNT(u.id), 0) as registros,
                DAYNAME(ds.fecha) as dia_semana
            FROM date_series ds
            LEFT JOIN users u ON DATE(u.creado_en) = ds.fecha
            GROUP BY ds.fecha
            ORDER BY ds.fecha
        ");
        $stmt->execute([$period - 1]);
        $results = $stmt->fetchAll(PDO::FETCH_ASSOC);

        $labels = [];
        $data = [];

        foreach ($results as $row) {
            $fecha = new DateTime($row['fecha']);
            $labels[] = $fecha->format('d/m');
            $data[] = (int)$row['registros'];
        }

        return [
            'labels' => $labels,
            'data' => $data
        ];
    }

    private function getValidacionesChart() {
        $stmt = $this->pdo->prepare("
            SELECT 
                resultado,
                COUNT(*) as total
            FROM validation_logs 
            WHERE timestamp >= DATE_SUB(NOW(), INTERVAL 30 DAY)
            GROUP BY resultado
        ");
        $stmt->execute();
        $results = $stmt->fetchAll(PDO::FETCH_ASSOC);

        $validaciones = [
            'exitosas' => 0,
            'fallidas' => 0,
            'error' => 0
        ];

        foreach ($results as $row) {
            switch ($row['resultado']) {
                case 'exitoso':
                    $validaciones['exitosas'] = (int)$row['total'];
                    break;
                case 'no_encontrado':
                    $validaciones['fallidas'] = (int)$row['total'];
                    break;
                case 'error':
                    $validaciones['error'] = (int)$row['total'];
                    break;
            }
        }

        return $validaciones;
    }

    private function getHorarios() {
        $stmt = $this->pdo->query("
            SELECT 
                HOUR(timestamp) as hora,
                COUNT(*) as total
            FROM validation_logs 
            WHERE timestamp >= DATE_SUB(NOW(), INTERVAL 7 DAY)
            AND resultado = 'exitoso'
            GROUP BY HOUR(timestamp)
            ORDER BY hora
        ");
        $results = $stmt->fetchAll(PDO::FETCH_ASSOC);

        $horarios = [];
        for ($i = 0; $i < 24; $i++) {
            $horarios[$i] = 0;
        }

        foreach ($results as $row) {
            $horarios[(int)$row['hora']] = (int)$row['total'];
        }

        return [
            'labels' => array_keys($horarios),
            'data' => array_values($horarios)
        ];
    }

    private function getRecentActivity() {
        // Logs administrativos recientes
        $stmt = $this->pdo->prepare("
            SELECT 
                al.*,
                a.nombre as admin_nombre,
                CASE 
                    WHEN al.accion = 'login_admin' THEN CONCAT('Inicio de sesión: ', a.nombre)
                    WHEN al.accion = 'usuario_creado' THEN 'Nuevo cliente registrado'
                    WHEN al.accion = 'usuario_actualizado' THEN 'Cliente actualizado'
                    WHEN al.accion = 'usuario_eliminado' THEN 'Cliente eliminado'
                    WHEN al.accion = 'backup_creado' THEN 'Backup de base de datos creado'
                    WHEN al.accion = 'config_actualizada' THEN 'Configuración actualizada'
                    ELSE al.accion
                END as descripcion
            FROM admin_logs al
            LEFT JOIN admins a ON al.admin_id = a.id
            ORDER BY al.timestamp DESC
            LIMIT 10
        ");
        $stmt->execute();
        $logs = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // Últimos clientes registrados
        $stmt = $this->pdo->prepare("
            SELECT 
                id,
                nombre,
                email,
                selfie_path,
                creado_en
            FROM users 
            ORDER BY creado_en DESC 
            LIMIT 8
        ");
        $stmt->execute();
        $clients = $stmt->fetchAll(PDO::FETCH_ASSOC);

        // Añadir tipo de actividad a los logs
        foreach ($logs as &$log) {
            switch ($log['accion']) {
                case 'login_admin':
                    $log['type'] = 'info';
                    break;
                case 'usuario_creado':
                    $log['type'] = 'success';
                    break;
                case 'usuario_eliminado':
                    $log['type'] = 'danger';
                    break;
                case 'backup_creado':
                    $log['type'] = 'success';
                    break;
                default:
                    $log['type'] = 'info';
            }
        }

        return [
            'logs' => $logs,
            'clients' => $clients
        ];
    }

    private function getSystemStatus() {
        $status = [];

        // Estado de la base de datos
        try {
            $this->pdo->query("SELECT 1");
            $status['db_connected'] = true;
        } catch (Exception $e) {
            $status['db_connected'] = false;
        }

        // Espacio usado por las imágenes
        $uploadsDir = __DIR__ . '/../../uploads/selfies/';
        $status['storage_used'] = $this->getDirectorySize($uploadsDir);

        // Último backup
        $stmt = $this->pdo->prepare("
            SELECT timestamp 
            FROM admin_logs 
            WHERE accion = 'backup_creado' 
            ORDER BY timestamp DESC 
            LIMIT 1
        ");
        $stmt->execute();
        $lastBackup = $stmt->fetch(PDO::FETCH_ASSOC);
        $status['last_backup'] = $lastBackup ? $lastBackup['timestamp'] : null;

        // Estadísticas de rendimiento
        $status['total_images'] = $this->getTotalImages();
        $status['avg_response_time'] = $this->getAverageResponseTime();

        // Verificar integridad de descriptores
        $status['descriptor_integrity'] = $this->checkDescriptorIntegrity();

        return $status;
    }

    private function getDirectorySize($directory) {
        $size = 0;
        
        if (!is_dir($directory)) {
            return 0;
        }
        
        foreach (new RecursiveIteratorIterator(new RecursiveDirectoryIterator($directory)) as $file) {
            $size += $file->getSize();
        }
        
        return round($size / 1024 / 1024, 2); // MB
    }

    private function getTotalImages() {
        $uploadsDir = __DIR__ . '/../../uploads/selfies/';
        
        if (!is_dir($uploadsDir)) {
            return 0;
        }
        
        $files = glob($uploadsDir . '*.{jpg,jpeg,png}', GLOB_BRACE);
        return count($files);
    }

    private function getAverageResponseTime() {
        // Simulación - en un entorno real, esto vendría de logs de servidor
        return rand(50, 200) . 'ms';
    }

    private function checkDescriptorIntegrity() {
        try {
            $stmt = $this->pdo->query("
                SELECT COUNT(*) as total,
                COUNT(CASE WHEN JSON_VALID(descriptor) THEN 1 END) as valid
                FROM users
            ");
            $result = $stmt->fetch(PDO::FETCH_ASSOC);
            
            $total = (int)$result['total'];
            $valid = (int)$result['valid'];
            
            return $total > 0 ? ($valid / $total) * 100 : 100;
            
        } catch (Exception $e) {
            return 0;
        }
    }

    private function logActivity($accion, $tabla_afectada = null, $registro_id = null, $detalles = []) {
        try {
            $stmt = $this->pdo->prepare("
                INSERT INTO admin_logs (admin_id, accion, tabla_afectada, registro_id, detalles, ip_address, user_agent)
                VALUES (?, ?, ?, ?, ?, ?, ?)
            ");
            
            $stmt->execute([
                $this->admin_id,
                $accion,
                $tabla_afectada,
                $registro_id,
                json_encode($detalles),
                $_SERVER['REMOTE_ADDR'] ?? '',
                $_SERVER['HTTP_USER_AGENT'] ?? ''
            ]);
        } catch (Exception $e) {
            // Log error pero no fallar la operación principal
            error_log("Error logging activity: " . $e->getMessage());
        }
    }

    // Método para exportar datos del dashboard
    public function exportDashboardData($format = 'json') {
        $data = $this->getDashboardData();
        
        switch ($format) {
            case 'csv':
                return $this->exportToCSV($data);
            case 'json':
            default:
                return $data;
        }
    }

    private function exportToCSV($data) {
        $output = fopen('php://temp', 'r+');
        
        // Headers
        fputcsv($output, [
            'Métrica',
            'Valor',
            'Fecha Generación'
        ]);
        
        // Stats data
        foreach ($data['stats'] as $key => $value) {
            fputcsv($output, [
                str_replace('_', ' ', ucfirst($key)),
                $value,
                date('Y-m-d H:i:s')
            ]);
        }
        
        rewind($output);
        $csv = stream_get_contents($output);
        fclose($output);
        
        return $csv;
    }

    // Método para obtener configuraciones del sistema
    public function getSystemConfig() {
        $stmt = $this->pdo->query("
            SELECT config_key, config_value, descripcion, tipo, categoria
            FROM system_config
            ORDER BY categoria, config_key
        ");
        
        return $stmt->fetchAll(PDO::FETCH_ASSOC);
    }

    // Método para limpiar datos antiguos
    public function cleanupOldData() {
        try {
            // Ejecutar procedimiento de limpieza
            $this->pdo->exec("CALL CleanOldLogs()");
            
            $this->logActivity('cleanup_executed', null, null, [
                'tipo' => 'limpieza_automatica',
                'timestamp' => date('Y-m-d H:i:s')
            ]);
            
            return ['success' => true, 'message' => 'Limpieza completada'];
            
        } catch (Exception $e) {
            return ['success' => false, 'message' => $e->getMessage()];
        }
    }
}

// Procesamiento de la petición
try {
    $dashboard = new DashboardAPI($pdo);
    
    // Determinar acción
    $action = $_GET['action'] ?? 'dashboard';
    
    switch ($action) {
        case 'dashboard':
            $result = $dashboard->getDashboardData();
            break;
            
        case 'export':
            $format = $_GET['format'] ?? 'json';
            $result = $dashboard->exportDashboardData($format);
            
            if ($format === 'csv') {
                header('Content-Type: text/csv');
                header('Content-Disposition: attachment; filename="dashboard_' . date('Y-m-d') . '.csv"');
                echo $result;
                exit;
            }
            break;
            
        case 'config':
            $result = ['success' => true, 'data' => $dashboard->getSystemConfig()];
            break;
            
        case 'cleanup':
            $result = $dashboard->cleanupOldData();
            break;
            
        default:
            $result = ['success' => false, 'message' => 'Acción no válida'];
    }
    
    echo json_encode($result, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Error interno del servidor',
        'debug' => $e->getMessage()
    ]);
}
?>