<?php
// /admin/api/clientes/stats.php
session_start();
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { exit; }

// Verificar autenticación
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'No autorizado']);
    exit;
}

require_once __DIR__ . '/../../../bd/bd.php';

try {
    // Estadísticas básicas
    $stats = [];
    
    // Total de clientes
    $stmt = $pdo->query("SELECT COUNT(*) as total FROM users");
    $stats['total'] = (int)$stmt->fetch()['total'];
    
    // Clientes registrados hoy
    $stmt = $pdo->query("SELECT COUNT(*) as hoy FROM users WHERE DATE(creado_en) = CURDATE()");
    $stats['hoy'] = (int)$stmt->fetch()['hoy'];
    
    // Clientes activos (validados en los últimos 30 días)
    $stmt = $pdo->query("
        SELECT COUNT(DISTINCT user_id) as activos 
        FROM validation_logs 
        WHERE timestamp >= DATE_SUB(NOW(), INTERVAL 30 DAY)
        AND resultado = 'exitoso'
        AND user_id IS NOT NULL
    ");
    $stats['activos'] = (int)$stmt->fetch()['activos'];
    
    // Posibles duplicados (nombres muy similares)
    $stmt = $pdo->query("
        SELECT COUNT(*) as duplicados
        FROM (
            SELECT nombre, COUNT(*) 
            FROM users 
            GROUP BY SOUNDEX(nombre) 
            HAVING COUNT(*) > 1
        ) as dups
    ");
    $stats['duplicados'] = (int)$stmt->fetch()['duplicados'];
    
    // Estadísticas adicionales
    $stmt = $pdo->query("SELECT COUNT(*) as semana FROM users WHERE creado_en >= DATE_SUB(NOW(), INTERVAL 7 DAY)");
    $stats['semana'] = (int)$stmt->fetch()['semana'];
    
    $stmt = $pdo->query("SELECT COUNT(*) as mes FROM users WHERE creado_en >= DATE_SUB(NOW(), INTERVAL 30 DAY)");
    $stats['mes'] = (int)$stmt->fetch()['mes'];
    
    // Distribución por dominio de email
    $stmt = $pdo->query("
        SELECT 
            SUBSTRING_INDEX(email, '@', -1) as dominio,
            COUNT(*) as total
        FROM users 
        WHERE email IS NOT NULL AND email != ''
        GROUP BY dominio
        ORDER BY total DESC
        LIMIT 5
    ");
    $stats['dominios'] = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo json_encode([
        'success' => true,
        'stats' => $stats
    ]);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Error obteniendo estadísticas',
        'debug' => $e->getMessage()
    ]);
}
?>

---

<?php
// /admin/api/clientes/list.php
session_start();
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { exit; }

// Verificar autenticación
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'No autorizado']);
    exit;
}

require_once __DIR__ . '/../../../bd/bd.php';

try {
    // Parámetros de filtrado y paginación
    $page = (int)($_GET['page'] ?? 1);
    $limit = (int)($_GET['limit'] ?? 25);
    $search = $_GET['search'] ?? '';
    $fecha_filtro = $_GET['fecha'] ?? '';
    $estado_filtro = $_GET['estado'] ?? '';
    $orden = $_GET['orden'] ?? 'creado_desc';
    
    $offset = ($page - 1) * $limit;
    
    // Construir query base
    $where_conditions = [];
    $params = [];
    
    // Filtro de búsqueda
    if (!empty($search)) {
        $where_conditions[] = "(u.nombre LIKE ? OR u.email LIKE ? OR u.telefono LIKE ?)";
        $search_param = "%{$search}%";
        $params[] = $search_param;
        $params[] = $search_param;
        $params[] = $search_param;
    }
    
    // Filtro de fecha
    if (!empty($fecha_filtro)) {
        switch ($fecha_filtro) {
            case 'hoy':
                $where_conditions[] = "DATE(u.creado_en) = CURDATE()";
                break;
            case 'semana':
                $where_conditions[] = "u.creado_en >= DATE_SUB(NOW(), INTERVAL 7 DAY)";
                break;
            case 'mes':
                $where_conditions[] = "u.creado_en >= DATE_SUB(NOW(), INTERVAL 30 DAY)";
                break;
            case '3meses':
                $where_conditions[] = "u.creado_en >= DATE_SUB(NOW(), INTERVAL 90 DAY)";
                break;
            case 'ano':
                $where_conditions[] = "YEAR(u.creado_en) = YEAR(NOW())";
                break;
        }
    }
    
    // Construir WHERE clause
    $where_sql = '';
    if (!empty($where_conditions)) {
        $where_sql = 'WHERE ' . implode(' AND ', $where_conditions);
    }
    
    // Construir ORDER BY
    $order_sql = 'ORDER BY ';
    switch ($orden) {
        case 'nombre_asc':
            $order_sql .= 'u.nombre ASC';
            break;
        case 'nombre_desc':
            $order_sql .= 'u.nombre DESC';
            break;
        case 'creado_asc':
            $order_sql .= 'u.creado_en ASC';
            break;
        case 'creado_desc':
        default:
            $order_sql .= 'u.creado_en DESC';
            break;
    }
    
    // Query principal con última validación
    $sql = "
        SELECT 
            u.*,
            vl.ultima_validacion,
            vl.total_validaciones
        FROM users u
        LEFT JOIN (
            SELECT 
                user_id,
                MAX(timestamp) as ultima_validacion,
                COUNT(*) as total_validaciones
            FROM validation_logs 
            WHERE resultado = 'exitoso'
            GROUP BY user_id
        ) vl ON u.id = vl.user_id
        {$where_sql}
        {$order_sql}
        LIMIT ? OFFSET ?
    ";
    
    $params[] = $limit;
    $params[] = $offset;
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    $clients = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    // Contar total para paginación
    $count_sql = "
        SELECT COUNT(*) as total
        FROM users u
        {$where_sql}
    ";
    
    $count_params = array_slice($params, 0, -2); // Remover limit y offset
    $stmt = $pdo->prepare($count_sql);
    $stmt->execute($count_params);
    $total_records = (int)$stmt->fetch()['total'];
    
    // Procesar datos de clientes
    foreach ($clients as &$client) {
        // Determinar estado basado en actividad reciente
        if (empty($client['estado'])) {
            if ($client['ultima_validacion']) {
                $diff = time() - strtotime($client['ultima_validacion']);
                $client['estado'] = ($diff < 2592000) ? 'activo' : 'inactivo'; // 30 días
            } else {
                $client['estado'] = 'inactivo';
            }
        }
        
        // Limpiar datos sensibles
        unset($client['descriptor']);
    }
    
    echo json_encode([
        'success' => true,
        'data' => $clients,
        'pagination' => [
            'page' => $page,
            'limit' => $limit,
            'total' => $total_records,
            'pages' => ceil($total_records / $limit)
        ]
    ]);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Error obteniendo lista de clientes',
        'debug' => $e->getMessage()
    ]);
}
?>

---

<?php
// /admin/api/clientes/details.php
session_start();
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { exit; }

// Verificar autenticación
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'No autorizado']);
    exit;
}

require_once __DIR__ . '/../../../bd/bd.php';

try {
    $client_id = (int)($_GET['id'] ?? 0);
    
    if ($client_id <= 0) {
        throw new Exception('ID de cliente no válido');
    }
    
    // Obtener datos del cliente
    $stmt = $pdo->prepare("
        SELECT 
            u.*,
            COUNT(vl.id) as total_validaciones,
            MAX(vl.timestamp) as ultima_validacion,
            AVG(CASE WHEN vl.confianza IS NOT NULL THEN vl.confianza END) as confianza_promedio
        FROM users u
        LEFT JOIN validation_logs vl ON u.id = vl.user_id AND vl.resultado = 'exitoso'
        WHERE u.id = ?
        GROUP BY u.id
    ");
    
    $stmt->execute([$client_id]);
    $client = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$client) {
        throw new Exception('Cliente no encontrado');
    }
    
    // Obtener estadísticas adicionales
    $stmt = $pdo->prepare("
        SELECT 
            COUNT(*) as total_intentos,
            SUM(CASE WHEN resultado = 'exitoso' THEN 1 ELSE 0 END) as exitosos,
            SUM(CASE WHEN resultado = 'no_encontrado' THEN 1 ELSE 0 END) as fallidos,
            SUM(CASE WHEN resultado = 'error' THEN 1 ELSE 0 END) as errores
        FROM validation_logs 
        WHERE user_id = ?
    ");
    
    $stmt->execute([$client_id]);
    $stats = $stmt->fetch(PDO::FETCH_ASSOC);
    
    $client['estadisticas'] = $stats;
    
    // Verificar integridad del descriptor
    $client['descriptor_valido'] = false;
    if (!empty($client['descriptor'])) {
        $descriptor = json_decode($client['descriptor'], true);
        if (is_array($descriptor) && count($descriptor) >= 128) {
            $client['descriptor_valido'] = true;
        }
    }
    
    // No enviar el descriptor completo por seguridad
    unset($client['descriptor']);
    
    // Log de acceso
    $admin_id = $_SESSION['admin_id'] ?? null;
    $stmt = $pdo->prepare("
        INSERT INTO admin_logs (admin_id, accion, tabla_afectada, registro_id, detalles, ip_address)
        VALUES (?, 'ver_cliente', 'users', ?, ?, ?)
    ");
    $stmt->execute([
        $admin_id,
        $client_id,
        json_encode(['nombre' => $client['nombre']]),
        $_SERVER['REMOTE_ADDR'] ?? ''
    ]);
    
    echo json_encode([
        'success' => true,
        'data' => $client
    ]);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
?>

---

<?php
// /admin/api/clientes/history.php
session_start();
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: GET, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { exit; }

// Verificar autenticación
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'No autorizado']);
    exit;
}

require_once __DIR__ . '/../../../bd/bd.php';

try {
    $client_id = (int)($_GET['id'] ?? 0);
    $limit = (int)($_GET['limit'] ?? 20);
    
    if ($client_id <= 0) {
        throw new Exception('ID de cliente no válido');
    }
    
    $stmt = $pdo->prepare("
        SELECT 
            resultado,
            confianza,
            timestamp,
            ip_address
        FROM validation_logs 
        WHERE user_id = ?
        ORDER BY timestamp DESC
        LIMIT ?
    ");
    
    $stmt->execute([$client_id, $limit]);
    $history = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    echo json_encode([
        'success' => true,
        'data' => $history
    ]);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
?>

---

<?php
// /admin/api/clientes/create.php
session_start();
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { exit; }

// Verificar autenticación
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'No autorizado']);
    exit;
}

require_once __DIR__ . '/../../../bd/bd.php';

try {
    // Validar datos requeridos
    if (empty($_POST['nombre'])) {
        throw new Exception('El nombre es requerido');
    }
    
    if (!isset($_FILES['file']) || $_FILES['file']['error'] !== UPLOAD_ERR_OK) {
        throw new Exception('La imagen es requerida');
    }
    
    $nombre = trim($_POST['nombre']);
    $email = trim($_POST['email'] ?? '');
    $telefono = trim($_POST['telefono'] ?? '');
    $procesarDescriptor = isset($_POST['procesarDescriptor']) && $_POST['procesarDescriptor'] === 'true';
    
    // Validaciones
    if (strlen($nombre) < 2) {
        throw new Exception('El nombre debe tener al menos 2 caracteres');
    }
    
    if (!empty($email) && !filter_var($email, FILTER_VALIDATE_EMAIL)) {
        throw new Exception('Email no válido');
    }
    
    if (!empty($telefono) && strlen($telefono) < 8) {
        throw new Exception('Teléfono debe tener al menos 8 dígitos');
    }
    
    // Verificar duplicados
    if (!empty($email)) {
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM users WHERE email = ?");
        $stmt->execute([$email]);
        if ($stmt->fetchColumn() > 0) {
            throw new Exception('Ya existe un cliente con este email');
        }
    }
    
    // Procesar imagen
    $file = $_FILES['file'];
    $allowedTypes = ['image/jpeg', 'image/png'];
    $maxSize = 5 * 1024 * 1024; // 5MB
    
    if (!in_array($file['type'], $allowedTypes)) {
        throw new Exception('Solo se permiten imágenes JPG y PNG');
    }
    
    if ($file['size'] > $maxSize) {
        throw new Exception('La imagen es demasiado grande (máximo 5MB)');
    }
    
    // Verificar que es una imagen válida
    $imageInfo = getimagesize($file['tmp_name']);
    if ($imageInfo === false) {
        throw new Exception('El archivo no es una imagen válida');
    }
    
    // Generar nombre único para el archivo
    $ext = $file['type'] === 'image/jpeg' ? 'jpg' : 'png';
    $fileName = 'admin_' . uniqid('', true) . '.' . $ext;
    $uploadPath = __DIR__ . '/../../../uploads/selfies/' . $fileName;
    $relativePath = 'uploads/selfies/' . $fileName;
    
    // Crear directorio si no existe
    $uploadDir = dirname($uploadPath);
    if (!is_dir($uploadDir)) {
        mkdir($uploadDir, 0775, true);
    }
    
    // Mover archivo
    if (!move_uploaded_file($file['tmp_name'], $uploadPath)) {
        throw new Exception('Error guardando la imagen');
    }
    
    // Generar descriptor facial (simulado por ahora)
    $descriptor = null;
    if ($procesarDescriptor) {
        // En un entorno real, aquí se procesaría la imagen con face-api.js
        // Por ahora generamos un descriptor simulado
        $descriptor = array_fill(0, 128, 0);
        for ($i = 0; $i < 128; $i++) {
            $descriptor[$i] = (mt_rand(-1000, 1000) / 1000);
        }
    }
    
    // Insertar en base de datos
    $stmt = $pdo->prepare("
        INSERT INTO users (nombre, telefono, email, selfie_path, descriptor, creado_en)
        VALUES (?, ?, ?, ?, ?, NOW())
    ");
    
    $stmt->execute([
        $nombre,
        $telefono ?: null,
        $email ?: null,
        $relativePath,
        $descriptor ? json_encode($descriptor) : null
    ]);
    
    $clientId = $pdo->lastInsertId();
    
    // Log de actividad
    $admin_id = $_SESSION['admin_id'] ?? null;
    $stmt = $pdo->prepare("
        INSERT INTO admin_logs (admin_id, accion, tabla_afectada, registro_id, detalles, ip_address)
        VALUES (?, 'usuario_creado_admin', 'users', ?, ?, ?)
    ");
    $stmt->execute([
        $admin_id,
        $clientId,
        json_encode([
            'nombre' => $nombre,
            'email' => $email,
            'telefono' => $telefono,
            'descriptor_generado' => $procesarDescriptor
        ]),
        $_SERVER['REMOTE_ADDR'] ?? ''
    ]);
    
    echo json_encode([
        'success' => true,
        'message' => 'Cliente creado exitosamente',
        'data' => [
            'id' => $clientId,
            'nombre' => $nombre,
            'descriptor_generado' => $procesarDescriptor
        ]
    ]);
    
} catch (Exception $e) {
    // Limpiar archivo si se subió pero falló la BD
    if (isset($uploadPath) && file_exists($uploadPath)) {
        unlink($uploadPath);
    }
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
?>

---

<?php
// /admin/api/clientes/update.php
session_start();
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: POST, PUT, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { exit; }

// Verificar autenticación
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'No autorizado']);
    exit;
}

require_once __DIR__ . '/../../../bd/bd.php';

try {
    // Obtener datos JSON
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        throw new Exception('Datos no válidos');
    }
    
    $client_id = (int)($input['id'] ?? 0);
    $nombre = trim($input['nombre'] ?? '');
    $email = trim($input['email'] ?? '');
    $telefono = trim($input['telefono'] ?? '');
    $estado = $input['estado'] ?? 'activo';
    
    if ($client_id <= 0) {
        throw new Exception('ID de cliente no válido');
    }
    
    if (empty($nombre)) {
        throw new Exception('El nombre es requerido');
    }
    
    // Validaciones
    if (!empty($email) && !filter_var($email, FILTER_VALIDATE_EMAIL)) {
        throw new Exception('Email no válido');
    }
    
    if (!in_array($estado, ['activo', 'inactivo', 'bloqueado'])) {
        throw new Exception('Estado no válido');
    }
    
    // Verificar que el cliente existe
    $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
    $stmt->execute([$client_id]);
    $currentClient = $stmt->fetch(PDO::FETCH_ASSOC);
    
    if (!$currentClient) {
        throw new Exception('Cliente no encontrado');
    }
    
    // Verificar duplicados de email
    if (!empty($email) && $email !== $currentClient['email']) {
        $stmt = $pdo->prepare("SELECT COUNT(*) FROM users WHERE email = ? AND id != ?");
        $stmt->execute([$email, $client_id]);
        if ($stmt->fetchColumn() > 0) {
            throw new Exception('Ya existe otro cliente con este email');
        }
    }
    
    // Configurar variables de sesión para el trigger
    $admin_id = $_SESSION['admin_id'] ?? null;
    $pdo->exec("SET @current_admin_id = " . ($admin_id ? $admin_id : 'NULL'));
    $pdo->exec("SET @current_admin_ip = '" . ($_SERVER['REMOTE_ADDR'] ?? '') . "'");
    
    // Actualizar cliente
    $stmt = $pdo->prepare("
        UPDATE users 
        SET nombre = ?, email = ?, telefono = ?, estado = ?
        WHERE id = ?
    ");
    
    $stmt->execute([
        $nombre,
        $email ?: null,
        $telefono ?: null,
        $estado,
        $client_id
    ]);
    
    if ($stmt->rowCount() === 0) {
        throw new Exception('No se realizaron cambios');
    }
    
    echo json_encode([
        'success' => true,
        'message' => 'Cliente actualizado exitosamente'
    ]);
    
} catch (Exception $e) {
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
?>

---

<?php
// /admin/api/clientes/delete.php
session_start();
header('Content-Type: application/json; charset=utf-8');
header('Access-Control-Allow-Origin: *');
header('Access-Control-Allow-Methods: DELETE, POST, OPTIONS');
header('Access-Control-Allow-Headers: Content-Type');

if ($_SERVER['REQUEST_METHOD'] === 'OPTIONS') { exit; }

// Verificar autenticación
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    http_response_code(401);
    echo json_encode(['success' => false, 'message' => 'No autorizado']);
    exit;
}

require_once __DIR__ . '/../../../bd/bd.php';

try {
    // Obtener ID(s) a eliminar
    $input = json_decode(file_get_contents('php://input'), true);
    
    if (!$input) {
        // Si no es JSON, intentar GET
        $client_id = (int)($_GET['id'] ?? 0);
        $client_ids = $client_id > 0 ? [$client_id] : [];
    } else {
        $client_ids = $input['ids'] ?? [];
        if (!is_array($client_ids)) {
            $client_ids = isset($input['id']) ? [(int)$input['id']] : [];
        }
    }
    
    if (empty($client_ids)) {
        throw new Exception('No se especificaron clientes para eliminar');
    }
    
    // Validar IDs
    $client_ids = array_filter(array_map('intval', $client_ids));
    if (empty($client_ids)) {
        throw new Exception('IDs de cliente no válidos');
    }
    
    $deletedCount = 0;
    $deletedFiles = [];
    
    // Configurar variables para triggers
    $admin_id = $_SESSION['admin_id'] ?? null;
    $pdo->exec("SET @current_admin_id = " . ($admin_id ? $admin_id : 'NULL'));
    $pdo->exec("SET @current_admin_ip = '" . ($_SERVER['REMOTE_ADDR'] ?? '') . "'");
    
    $pdo->beginTransaction();
    
    foreach ($client_ids as $client_id) {
        // Obtener datos del cliente antes de eliminar
        $stmt = $pdo->prepare("SELECT * FROM users WHERE id = ?");
        $stmt->execute([$client_id]);
        $client = $stmt->fetch(PDO::FETCH_ASSOC);
        
        if (!$client) {
            continue; // Cliente no existe, saltar
        }
        
        // Eliminar registros relacionados
        $stmt = $pdo->prepare("DELETE FROM validation_logs WHERE user_id = ?");
        $stmt->execute([$client_id]);
        
        // Eliminar cliente
        $stmt = $pdo->prepare("DELETE FROM users WHERE id = ?");
        $stmt->execute([$client_id]);
        
        if ($stmt->rowCount() > 0) {
            $deletedCount++;
            
            // Marcar archivo para eliminación
            if (!empty($client['selfie_path'])) {
                $filePath = __DIR__ . '/../../../' . $client['selfie_path'];
                if (file_exists($filePath)) {
                    $deletedFiles[] = $filePath;
                }
            }
        }
    }
    
    $pdo->commit();
    
    // Eliminar archivos de imágenes
    foreach ($deletedFiles as $filePath) {
        if (file_exists($filePath)) {
            unlink($filePath);
        }
    }
    
    if ($deletedCount === 0) {
        throw new Exception('No se eliminaron clientes');
    }
    
    echo json_encode([
        'success' => true,
        'message' => "Se eliminaron {$deletedCount} cliente(s) exitosamente",
        'deleted_count' => $deletedCount
    ]);
    
} catch (Exception $e) {
    if ($pdo->inTransaction()) {
        $pdo->rollback();
    }
    
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => $e->getMessage()
    ]);
}
?>

---

<?php
// /admin/api/clientes/export.php
session_start();
header('Content-Type: text/csv; charset=utf-8');
header('Content-Disposition: attachment; filename="clientes_' . date('Y-m-d_H-i-s') . '.csv"');
header('Access-Control-Allow-Origin: *');

// Verificar autenticación
if (!isset($_SESSION['admin_logged_in']) || !$_SESSION['admin_logged_in']) {
    http_response_code(401);
    echo "No autorizado";
    exit;
}

require_once __DIR__ . '/../../../bd/bd.php';

try {
    // Obtener parámetros
    $format = $_GET['format'] ?? 'csv';
    $include_stats = isset($_GET['include_stats']) && $_GET['include_stats'] === '1';
    $date_from = $_GET['date_from'] ?? '';
    $date_to = $_GET['date_to'] ?? '';
    
    // Construir query
    $where_conditions = [];
    $params = [];
    
    if (!empty($date_from)) {
        $where_conditions[] = "u.creado_en >= ?";
        $params[] = $date_from . ' 00:00:00';
    }
    
    if (!empty($date_to)) {
        $where_conditions[] = "u.creado_en <= ?";
        $params[] = $date_to . ' 23:59:59';
    }
    
    $where_sql = '';
    if (!empty($where_conditions)) {
        $where_sql = 'WHERE ' . implode(' AND ', $where_conditions);
    }
    
    $sql = "
        SELECT 
            u.id,
            u.nombre,
            u.email,
            u.telefono,
            u.creado_en,
            u.estado,
            COUNT(vl.id) as total_validaciones,
            MAX(vl.timestamp) as ultima_validacion
        FROM users u
        LEFT JOIN validation_logs vl ON u.id = vl.user_id AND vl.resultado = 'exitoso'
        {$where_sql}
        GROUP BY u.id
        ORDER BY u.creado_en DESC
    ";
    
    $stmt = $pdo->prepare($sql);
    $stmt->execute($params);
    $clients = $stmt->fetchAll(PDO::FETCH_ASSOC);
    
    if ($format === 'json') {
        header('Content-Type: application/json; charset=utf-8');
        header('Content-Disposition: attachment; filename="clientes_' . date('Y-m-d_H-i-s') . '.json"');
        echo json_encode($clients, JSON_UNESCAPED_UNICODE | JSON_PRETTY_PRINT);
        exit;
    }
    
    // Generar CSV
    $output = fopen('php://output', 'w');
    
    // UTF-8 BOM para Excel
    fputs($output, "\xEF\xBB\xBF");
    
    // Headers CSV
    $headers = [
        'ID',
        'Nombre',
        'Email',
        'Teléfono',
        'Fecha Registro',
        'Estado',
        'Total Validaciones',
        'Última Validación'
    ];
    
    fputcsv($output, $headers, ',', '"');
    
    // Datos
    foreach ($clients as $client) {
        $row = [
            $client['id'],
            $client['nombre'],
            $client['email'] ?? '',
            $client['telefono'] ?? '',
            $client['creado_en'],
            $client['estado'] ?? 'activo',
            $client['total_validaciones'] ?? 0,
            $client['ultima_validacion'] ?? ''
        ];
        
        fputcsv($output, $row, ',', '"');
    }
    
    fclose($output);
    
    // Log de exportación
    $admin_id = $_SESSION['admin_id'] ?? null;
    $stmt = $pdo->prepare("
        INSERT INTO admin_logs (admin_id, accion, tabla_afectada, detalles, ip_address)
        VALUES (?, 'exportar_clientes', 'users', ?, ?)
    ");
    $stmt->execute([
        $admin_id,
        json_encode([
            'formato' => $format,
            'total_registros' => count($clients),
            'filtros' => [
                'fecha_desde' => $date_from,
                'fecha_hasta' => $date_to
            ]
        ]),
        $_SERVER['REMOTE_ADDR'] ?? ''
    ]);
    
} catch (Exception $e) {
    header('Content-Type: application/json; charset=utf-8');
    header('Content-Disposition: inline');
    http_response_code(500);
    echo json_encode([
        'success' => false,
        'message' => 'Error en exportación: ' . $e->getMessage()
    ]);
}
?>